<?php
namespace Concrete\Package\SocialFeed\Block\EasySocialFeed;

use Concrete\Core\Block\BlockController;
use Core;

use \Abraham\TwitterOAuth\TwitterOAuth;
use \Instagram\InstagramClient;

class Controller extends BlockController
{
  protected $btTable = 'btEasySocialFeed';
  protected $btInterfaceWidth = "600";
  protected $btInterfaceHeight = "490";
  protected $btCacheBlockRecord = false;
  protected $btCacheBlockOutput = false;
  protected $btCacheBlockOutputOnPost = false;
  protected $btCacheBlockOutputForRegisteredUsers = false;
  protected $btDefaultSet = 'social';

  /**
  * Used for localization. If we want to localize the name/description we have to include this.
  */
  public function getBlockTypeDescription()
  {
    return t("Easily add your social media feed to the website.");
  }

  public function getBlockTypeName()
  {
    return t("Easy Social Feed");
  }

  public function get_data_ajax($bID = null, $account = 'all', $skip = 0){
    if(empty($bID))
      exit();

    $blockData = $this->getBlockData($bID)[0];
    if(empty($blockData))
    exit();

    if($account === 'all')
    $account = null;

    $updateNeeded = $this->isUpdateNeeded($blockData);

    if($updateNeeded){
      $this->updateFeedData($blockData);
      $this->getFeedDataFromDB($blockData, $account, $skip);
    }
    else{
      $this->getFeedDataFromDB($blockData, $account, $skip);
    }
  }

  public function getBlockData($bID){
    $db = \Database::connection();
    $data = $db->GetAll('SELECT * from btEasySocialFeed WHERE bID = ?', array($bID));
    return $data;
  }

  private function isUpdateNeeded($data){
    return !$data['using_cronjob'] && strtotime($data['last_update']) < strtotime("-". $data['cache_duration'] . " minutes");
  }

  public function initFacebookUpdate($blockData){
    $FBAppId = $blockData['facebook_key'];
    $FBAppSecret = $blockData['facebook_secret'];
    $FBPageId = $blockData['facebook_page_id'];
    $FBAccessToken = $blockData['facebook_access_token'];
    if(!empty($FBAppId) && !empty($FBAppSecret) && !empty($FBPageId) && !empty($FBAccessToken)){
      return $this->updateFacebook($blockData);
    }
  }

  public function initTwitterUpdate($blockData){
    $TAppId = $blockData['twitter_key'];
    $TAppSecret = $blockData['twitter_secret'];
    $TConsumerKey = $blockData['twitter_consumer_key'];
    $TConsumerSecret = $blockData['twitter_consumer_secret'];
    $TPageId = $blockData['twitter_page_id'];
    if(!empty($TAppId) && !empty($TAppSecret) && !empty($TConsumerKey) && !empty($TConsumerSecret) && !empty($TPageId)){
      return $this->updateTwitter($blockData);
    }
  }

  public function initInstagramUpdate($blockData){
    $IAppKey = $blockData['instagram_key'];
    $IAppSecret = $blockData['instagram_secret'];
    $IAccessToken = $blockData['instagram_access_token'];
    $IPageId = $blockData['instagram_page_id'];
    if(!empty($IAppKey) && !empty($IAppSecret) && !empty($IAccessToken) && !empty($IPageId)){
      return $this->updateInstagram($blockData);
    }
  }

  public function initYoutubeUpdate($blockData){
    $YTKey = $blockData['youtube_key'];
    $YTChannelId = $blockData['youtube_channel_id'];
    if(!empty($YTKey) && !empty($YTChannelId)){
      return $this->updateYoutube($blockData);
    }
  }

  public function updateFeedData($blockData){

    //Check Facebook
    $FBLastId = $this->initFacebookUpdate($blockData);

    //Check Twitter
    $TLastId = $this->initTwitterUpdate($blockData);

    //Check Instagram
    $ILastId = $this->initInstagramUpdate($blockData);

    //Check Youtube
    $YTLastId = $this->initYoutubeUpdate($blockData);

  }


  private function getFeedDataFromDB($blockData, $socialNetwork = null, $skip = 0){
    $db = \Database::connection();
    $params = array();
    $limit = $blockData['posts_limit'];
    $QB = $db->createQueryBuilder();
    $QB->select('*')
    ->from('btEasySocialFeedEntries', 's')
    ->setFirstResult($skip)
    ->setMaxResults($limit)
    ->orderBy('s.published_at', 'DESC');

    switch($socialNetwork){
      case 'twitter':
        $QB->where('s.user_id = ?');
        array_push($params, $blockData['twitter_page_id']);
        if($blockData['twitter_exclude_replies']){
          $QB->andWhere('s.post_type IS NULL OR s.post_type != "reply"');
        }
        if($blockData['twitter_exclude_retweets']){
          $QB->andWhere('s.post_type IS NULL OR s.post_type != "retweet"');
        }
      break;
      case 'facebook':
        $QB->where('s.user_id = ?');
        array_push($params, $blockData['facebook_page_id']);
      break;
      case 'instagram':
        $QB->where('s.user_id = ?');
        array_push($params, $blockData['instagram_page_id']);
      break;
      case 'youtube':
        $QB->where('s.user_id = ?');
        array_push($params, $blockData['youtube_channel_id']);
      break;
      default:

        $QB->where('s.user_id IN (?,?,?,?)');
        array_push($params, $blockData['twitter_page_id'], $blockData['facebook_page_id'], $blockData['instagram_page_id'], $blockData['youtube_channel_id']);

        if($blockData['twitter_exclude_replies']){
          $QB->andWhere('s.post_type IS NULL OR s.post_type != "reply"');
        }
        if($blockData['twitter_exclude_retweets']){
          $QB->andWhere('s.post_type IS NULL OR s.post_type != "retweet"');
        }
      break;
    }


    if(!empty($socialNetwork)){
      $QB->andWhere('s.social_account = ?');
      array_push($params, $socialNetwork);
    }
    $result = $db->getAll($QB, $params);

    foreach($result as $key => $res){
      $datetime = new \DateTime($res['published_at']);
      $result[$key]['published_at'] = $datetime->format('c');
    }
    $this->sendJSON($result);
  }

  public function sendJson( $data, $error = null )
  {
    $ajax = Core::make( 'helper/ajax' );
    if ( !$error ) {
      $ajax->sendResult($data);
    }
    $ajax->sendError($data);
  }

  private function updateFacebook($blockData){
    $appId = $blockData['facebook_key'];
    $appSecret = $blockData['facebook_secret'];
    $accessToken = $blockData['facebook_access_token'];
    $pageId = $blockData['facebook_page_id'];
    $limit = $blockData['posts_limit'];
    $lastId = $blockData['facebook_last_id'];

    $posts = array();

    $fb = new \Facebook\Facebook([
        'app_id' => $appId,
        'app_secret' => $appSecret,
        'default_graph_version' => 'v3.2',
        'date_format' => 'U'
        ]
      );
    $fb->setDefaultAccessToken($accessToken);

    try {
      $profileRequest = $fb->get($pageId . '?fields=link,name,picture');
      $profile = $profileRequest->getDecodedBody();

      $postsRequest = $fb->get($pageId . '/feed?fields=message,created_time,permalink_url,object_id,full_picture,attachments&limit=100');
    }  catch(Facebook\Exceptions\FacebookResponseException $e) {
      die('Graph returned an error: ' . $e->getMessage());
    exit;
    } catch(Facebook\Exceptions\FacebookSDKException $e) {
      die('Facebook SDK returned an error: ' . $e->getMessage());
    }

    $decodedPosts = $postsRequest->getDecodedBody();

    foreach($decodedPosts['data'] as $key => $value){
      if($key === 0)
        $lastId = strtotime($value['created_time']);
      $d = strtotime($value['created_time']);
      $d = date('Y-m-d H:i:s', $d);

      array_push($posts, array(
        'id' => 'FB-'. $blockData['facebook_page_id'] . '-' . $value['id'],
        'text' => $value['message'],
        'type' => 'post',
        'name' => $profile['username'],
        'screen_name' => $profile['name'],
        'created_at' => $d,
        'profile_img' => $profile['picture']['data']['url'],
        'profile_url' => $profile['link'],
        'read_more_url' => $value['permalink_url'],
        'media' => (!empty($value['full_picture'])) ? $value['full_picture'] : null
      ));
    }

    $this->insertSocialData($blockData, $posts, 'facebook', $lastId);

    return $lastId;
  }


  private function updateTwitter($blockData){
    $accessToken = $blockData['twitter_key'];
    $accessSecret = $blockData['twitter_secret'];
    $consumerKey = $blockData['twitter_consumer_key'];
    $consumerSecret = $blockData['twitter_consumer_secret'];
    $excludeReplies = $blockData['twitter_exclude_replies'] ? true : false;
    $username = $blockData['twitter_page_id'];
    $limit = $blockData['posts_limit'];
    $displayImages = $blockData['display_images'];
    $lastId = $blockData['twitter_last_id'];
    $tweets = array();

    $connection = new TwitterOAuth($consumerKey, $consumerSecret, $accessToken, $accessSecret);

      $content = $connection->get("account/verify_credentials");

      $config = array(
        'count' => 200,
        'exclude_replies' => $excludeReplies,
        'screen_name' => $username,
        'include_rts' => true,
        'exclude_replies' => false
      );


      if(!empty($lastId))
      $config['since_id'] = $lastId;

      $statuses = $connection->get("statuses/user_timeline", $config);
      if(!empty($statuses->errors)){
        return $lastId;
      }



      foreach($statuses as $key => $value){
        $type = null;
        $d = \DateTime::createFromFormat('D M d H:i:s P Y', (string)$value->created_at);
        $d = $d->format('Y-m-d H:i:s');
        if($key === 0)
        $lastId = $value->id;
        if($value->retweeted)
          $type = 'retweet';
        if($value->in_reply_to_status_id)
          $type = 'reply';

        array_push($tweets, array(
          'id' => 'TW-'. $blockData['twitter_page_id'] . '-' . $value->id,
          'type' => $type,
          'text' => (!$value->retweeted) ? $value->text : $value->retweeted_status->text,
          'name' => $value->user->name,
          'screen_name' => $value->user->screen_name,
          'created_at' => $d,
          'profile_img' => $value->user->profile_image_url_https,
          'profile_url' => 'https://twitter.com/' . $value->user->screen_name,
          'read_more_url' => 'https://twitter.com/' . $value->user->screen_name . '/status/' . $value->id,
          'media' => (isset($value->entities) && isset($value->entities->media) && isset($value->entities->media[0])) ? $value->entities->media[0]->media_url_https: null
        ));
      }

      $this->insertSocialData($blockData, $tweets, 'twitter', $lastId);

    return $lastId;
  }

  private function updateInstagram($blockData){
    $appKey = $blockData['instagram_key'];
    $appSecret = $blockData['instagram_secret'];
    $accessToken = $blockData['instagram_access_token'];
    $limit = $blockData['posts_limit'];

    $config = array(
      'client_id' => $appKey,
      'client_secret' => $appSecret,
      'redirect_uri' => 'http://example.com'
    );
    $lastId = $blockData['instagram_last_id'];
    $posts = array();

    $ig = new InstagramClient( $config );
      $ig->set_access_token( $accessToken );
      $media = $ig->recentMedia( (int) $count = 100, $lastId, $blockData['instagram_page_id'] )->get_data();

      $media = json_decode($media);
      if(!empty($media->meta->error_message)){
        return $lastId;
      }

      $media = $media->data;
      foreach($media as $key => $value){
        if($key === 0)
        $lastId = $value->id;
        array_push($posts, array(
          'id' => 'IN-'. $blockData['instagram_page_id'] . '-' . $value->id,
          'text' => $value->caption ? $value->caption->text : $value->caption,
          'type' => 'post',
          'name' => $value->user->username,
          'screen_name' => $value->user->full_name,
          'created_at' => date('Y-m-d H:i:s', $value->created_time),
          'profile_img' => $value->user->profile_picture,
          'profile_url' => 'https://instagram.com/'. $value->user->username,
          'read_more_url' => $value->link,
          'media' => $value->images->standard_resolution->url
        ));
      }

      $this->insertSocialData($blockData, $posts, 'instagram', $lastId);

    return $lastId;
  }

  private function updateYoutube($blockData){
    $baseUrl = 'https://www.googleapis.com/youtube/v3/';
    $apiKey = $blockData['youtube_key'];
    $channelId = $blockData['youtube_channel_id'];
    $limit = $blockData['posts_limit'];
    $lastId = $blockData['youtube_last_id'];
    $videos = array();

    $params = [
      'id'=> $channelId,
      'part'=> 'contentDetails,snippet',
      'key'=> $apiKey
    ];

    $url = $baseUrl . 'channels?' . http_build_query($params);
      $json = json_decode(file_get_contents($url), true);


      $channelId = $json['items'][0]['id'];

      $channel = $json['items'][0]['snippet'];

      $playlist = $json['items'][0]['contentDetails']['relatedPlaylists']['uploads'];
      $params = [
        'part'=> 'snippet',
        'type' => 'video',
        'channelId' => $channelId,
        'order' => 'date',
        'maxResults' => '50',
        'key' => $apiKey
      ];

      if(!empty($lastId))
      $params['publishedAfter'] = $lastId;

      $url = $baseUrl . 'search?' . http_build_query($params);
      $json = json_decode(file_get_contents($url), true);

      foreach($json['items'] as $key => $video){
        $snippet = $video['snippet'];
        if($key === 0)
        $lastId = date("c", strtotime($snippet['publishedAt']));

        $d = strtotime($snippet['publishedAt']);
        $d = date('Y-m-d H:i:s', $d);
        array_push($videos, array(
          'id' => 'YT-'. $blockData['youtube_channel_id'] . '-' . $video['id']['videoId'],
          'text' => $snippet['title'],
          'type' => 'post',
          'name' => $snippet['channelTitle'],
          'screen_name' => $snippet['channelTitle'],
          'created_at' => $d,
          'profile_img' => $channel['thumbnails']['default']['url'],
          'profile_url' => 'https://www.youtube.com/channel/' . $channelId,
          'read_more_url' => 'https://www.youtube.com/watch?v=' . $video['id']['videoId'],
          'media' => $snippet['thumbnails']['high']['url']
        ));
      }

      $this->insertSocialData($blockData, $videos, 'youtube', $lastId);

    return $lastId;
  }

  private function updateCacheTimeAndLastID($bID, $type, $lastId){
    $db = \Database::connection();
    $QB = $db->createQueryBuilder();

    $QB->update('btEasySocialFeed', 'u')->set('u.last_update', '?');

    switch($type){
      case 'facebook':
        $QB->set('u.facebook_last_id', '?');
      break;
      case 'twitter':
        $QB->set('u.twitter_last_id', '?');
      break;
      case 'instagram':
        $QB->set('u.instagram_last_id', '?');
      break;
      case 'youtube':
        $QB->set('u.youtube_last_id', '?');
      break;
      default:

      break;
    }

    $QB->where('bID = ?');
    $db->execute($QB, array(date('Y-m-d H:i:s'), $lastId, $bID));

    }

    private function insertSocialData($blockData, $data, $social_account, $lastId){
      $db = \Database::connection();
      switch ($social_account){
        case 'twitter':
          $entryId = $blockData['twitter_page_id'];
        break;
        case 'facebook':
          $entryId = $blockData['facebook_page_id'];
        break;
        case 'instagram':
          $entryId = $blockData['instagram_page_id'];
        break;
        case 'youtube':
          $entryId = $blockData['youtube_channel_id'];
        break;
        default:

        break;
      }

      try {
        foreach($data as $key => $val){
            $db->Execute('SET NAMES \'utf8mb4\'');
            $db->Execute('INSERT INTO btEasySocialFeedEntries (
              id,
              post_type,
              user_id,
              social_account,
              profile_url,
              profile_img,
              `text`,
              name,
              screen_name,
              read_more_url,
              media,
              published_at
            ) VALUES (?,?,?,?,?,?,?,?,?,?,?,?)', array(
              $val['id'],
              $val['type'],
              $entryId,
              $social_account,
              $val['profile_url'],
              $val['profile_img'],
              $val['text'],
              $val['name'],
              $val['screen_name'],
              $val['read_more_url'],
              $val['media'],
              $val['created_at']
            ));
          }
        }
        catch(\Doctrine\DBAL\Exception\UniqueConstraintViolationException $e){

        }
        catch(\Doctrine\DBAL\Exception\DriverException $e){

        }
        catch(Exception $e){

        }
      $this->updateCacheTimeAndLastID($blockData['bID'], $social_account, $lastId);
    }

    public function getViewObject(){
      return $this->controllerView;
    }

    public function add(){
      $this->edit();
    }


    public function edit()
    {
      if(!$this->posts_limit){
        $this->set('posts_limit', 10);
      }
      if(!$this->width){
        $this->set('width', '100%');
      }
      if(!$this->cache_duration){
        $this->set('cache_duration', 10);
      }
      if(!$this->template){
        $this->set('template', 'default');
      }
      if(!$this->masonry_cols){
        $this->set('masonry_cols', 3);
      }
      if(!$this->masonry_cols_tablet){
        $this->set('masonry_cols_tablet', 2);
      }
      if(!$this->masonry_cols_mobile){
        $this->set('masonry_cols_mobile', 1);
      }
      if(!$this->default_tab){
        $this->set('default_tab', 'twitter');
      }
      if(!isset($this->display_images)){
        $this->set('display_images', 1);
      }
      if(!isset($this->twitter_exclude_replies)){
        $this->set('twitter_exclude_replies', 1);
      }
      if(!isset($this->twitter_exclude_retweets)){
        $this->set('twitter_exclude_retweets', 1);
      }
      if(!isset($this->update_on_save)){
        $this->set('update_on_save', 0);
      }

      if (!isset($this->locale)) {
        $this->set('locale', 'just now, %s seconds ago, 1 minute ago, %s minutes ago, 1 hour ago, %s hours ago, 1 day ago, %s days ago, 1 week ago, %s weeks ago, 1 month ago, %s months ago, 1 year ago, %s years ago');
      }
    }

    private function getUniqId($length = 10){
      return Core::make('helper/validation/identifier')->getString($length);
    }


    public function registerViewAssets($outputContent = '')
    {
      $this->requireAsset('javascript', 'jquery');
      $this->requireAsset('javascript', 'underscore');
      $this->requireAsset('easy_social_feed');
    }

    public function view()
    {
      $this->set('unique_block_id', $this->getUniqId());
      $this->set('facebook_valid', (
      !empty($this->facebook_key) &&
      !empty($this->facebook_secret) &&
      !empty($this->facebook_page_id) &&
      !empty($this->facebook_access_token)));

      $this->set('instagram_valid', (
      !empty($this->instagram_key) &&
      !empty($this->instagram_secret) &&
      !empty($this->instagram_access_token) &&
      !empty($this->instagram_page_id)));

      $this->set('twitter_valid', (
      !empty($this->twitter_key) &&
      !empty($this->twitter_secret) &&
      !empty($this->twitter_consumer_key) &&
      !empty($this->twitter_consumer_secret) &&
      !empty($this->twitter_page_id)));

      $this->set('youtube_valid', (
      !empty($this->youtube_key) &&
      !empty($this->youtube_channel_id)));

    }

    public function install($path){
      parent::install($path);
      $db = \Database::Connection();
      $db->execute('ALTER TABLE btEasySocialFeedEntries CHANGE `text` `text` VARCHAR(500) CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;');
    }

    public function validate($args)
    {
      $error = Core::make('helper/validation/error');

      if($args['posts_limit'] < 1){
        $error->add(t('Posts to display must be a positive number '));
      }

      if(!preg_match('/(\d*)(px|%|rem|em|vh|vw)/', $args['width'])){
        $error->add(t('Width must be px, %, rem, em, vh or vw'));
      }

      if(!empty($args['height']) && !preg_match('/(\d*)(px|%|rem|em|vh|vw)/', $args['height'])){
        $error->add(t('Height must be px, %, rem, em, vh or vw'));
      }

      if ($error->has()) {
        return $error;
      }

    }

    public function save($data)
    {
      $args['width'] = isset($data['width']) ? trim($data['width']) : '';
      $args['height'] = isset($data['height']) ? trim($data['height']) : '';

      $args['background_color'] = isset($data['background_color']) ? trim($data['background_color']) : '';
      $args['border_color'] = isset($data['border_color']) ? trim($data['border_color']) : '';
      $args['font_color'] = isset($data['font_color']) ? trim($data['font_color']) : '';
      $args['link_color'] = isset($data['link_color']) ? trim($data['link_color']) : '';
      $args['account_name_color'] = isset($data['account_name_color']) ? trim($data['account_name_color']) : '';

      $args['template'] = isset($data['template']) ? trim($data['template']) : 'default';
      $args['default_tab'] = isset($data['default_tab']) ? trim($data['default_tab']) : 'twitter';
      $args['posts_limit'] = isset($data['posts_limit']) ? trim($data['posts_limit']) : 5;
      $args['cache_duration'] = isset($data['cache_duration']) ? trim($data['cache_duration']) : 15;
      $args['masonry_cols'] = isset($data['masonry_cols']) ? trim($data['masonry_cols']) : 3;
      $args['masonry_cols_tablet'] = isset($data['masonry_cols_tablet']) ? trim($data['masonry_cols_tablet']) : 2;
      $args['masonry_cols_mobile'] = isset($data['masonry_cols_mobile']) ? trim($data['masonry_cols_mobile']) : 1;

      $args['facebook_key'] = isset($data['facebook_key']) ? trim($data['facebook_key']) : '';
      $args['facebook_secret'] = isset($data['facebook_secret']) ? trim($data['facebook_secret']) : '';
      $args['facebook_access_token'] = isset($data['facebook_access_token']) ? trim($data['facebook_access_token']) : '';
      $args['facebook_page_id'] = isset($data['facebook_page_id']) ? trim($data['facebook_page_id']) : '';

      $args['instagram_key'] = isset($data['instagram_key']) ? trim($data['instagram_key']) : '';
      $args['instagram_secret'] = isset($data['instagram_secret']) ? trim($data['instagram_secret']) : '';
      $args['instagram_access_token'] = isset($data['instagram_access_token']) ? trim($data['instagram_access_token']) : '';
      $args['instagram_page_id'] = isset($data['instagram_page_id']) ? trim($data['instagram_page_id']) : '';

      $args['twitter_key'] = isset($data['twitter_key']) ? trim($data['twitter_key']) : '';
      $args['twitter_secret'] = isset($data['twitter_secret']) ? trim($data['twitter_secret']) : '';
      $args['twitter_consumer_key'] = isset($data['twitter_consumer_key']) ? trim($data['twitter_consumer_key']) : '';
      $args['twitter_consumer_secret'] = isset($data['twitter_consumer_secret']) ? trim($data['twitter_consumer_secret']) : '';
      $args['twitter_page_id'] = isset($data['twitter_page_id']) ? trim($data['twitter_page_id']) : '';

      $args['youtube_key'] = isset($data['youtube_key']) ? trim($data['youtube_key']) : '';
      $args['youtube_channel_id'] = isset($data['youtube_channel_id']) ? trim($data['youtube_channel_id']) : '';

      $args['facebook_last_id'] = isset($data['facebook_last_id']) ? trim($data['facebook_last_id']) : '';
      $args['twitter_last_id'] = isset($data['twitter_last_id']) ? trim($data['twitter_last_id']) : '';
      $args['instagram_last_id'] = isset($data['instagram_last_id']) ? trim($data['instagram_last_id']) : '';
      $args['youtube_last_id'] = isset($data['youtube_last_id']) ? trim($data['youtube_last_id']) : '';

      $args['display_images'] = $data['display_images'] ? 1 : 0;
      $args['load_more'] = $data['load_more'] ? 1 : 0;
      $args['twitter_exclude_replies'] = $data['twitter_exclude_replies'] ? 1 : 0;
      $args['twitter_exclude_retweets'] = $data['twitter_exclude_retweets'] ? 1 : 0;
      $args['using_cronjob'] = $data['using_cronjob'] ? 1 : 0;
      $args['update_on_save'] = $data['update_on_save'] ? 1 : 0;
      $args['locale'] = isset($data['locale']) ? $data['locale'] : '';
      parent::save($args);

      if($args['update_on_save']){
        //Update feed
        $blockData = $this->getBlockData($this->bID);
        $this->updateFeedData($blockData[0]);
      }
    }
  }
