<?php

/**
 * ab_google_maps package for Concrete5
 * Copyright Copyright 2018-2019, Alex Borisov
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * @author Alex Borisov <linuxoidoz@gmail.com>
 * @package Concrete\Package\ab_google_maps
 */

namespace Concrete\Package\AbGoogleMaps\Block\AbGoogleMaps;

use Concrete\Core\Block\BlockController;
use Concrete\Core\File\File;
use Concrete\Core\Package\Package;

class Controller extends BlockController
{
    
    protected $btWrapperClass = 'ccm-ui';
    protected $btInterfaceWidth = "700";
    protected $btInterfaceHeight = "600";
    protected $btTable = 'btAbGoogleMaps';
    protected $btDefaultSet = 'multimedia';
    
    public function getBlockTypeDescription() 
    {
        return t("Add Free Google Maps block");
    }

    public function getBlockTypeName() 
    {
        return t("Free Google Maps");
    }

    public function registerViewAssets($outputContent = '')
    {
        $this->requireAsset('javascript', 'jquery');
        
        $html = $this->app->make('helper/html');
        if ($this->google_api_key != '') {
            $this->addHeaderItem($html->javascript("https://maps.googleapis.com/maps/api/js?key=" . $this->google_api_key . "&callback=InitGoogleMap"));
        }
        if ($this->locationiq_api_key != '') {
            $this->addHeaderItem($html->css("https://unpkg.com/leaflet@1.5.1/dist/leaflet.css"));
            $this->addHeaderItem($html->javascript("https://unpkg.com/leaflet@1.5.1/dist/leaflet.js"));
            $this->addHeaderItem($html->javascript("https://tiles.unwiredmaps.com/js/leaflet-unwired.js"));
        }
    }
    
    public function on_start() {
        $this->set('app', $this->app);
        
        $pkg = Package::getByHandle('ab_google_maps');
        $img_url = $pkg->getRelativePath() . '/blocks/ab_google_maps/images';
        $this->set("img_url", $img_url);
    }
    
    public function view() 
    {
        $name = $this->show_name ? $this->name : $this->location;
        
        $picture = '';
        if ($this->getPictureID() > 0) {
            $picture = $this->getPictureObject()->getVersion()->getRelativePath();
        }
        
        $data = array(
            'google_api_key' => $this->google_api_key,
            'latitude' => $this->latitude,
            'longitude' => $this->longitude,
            'zoom' => $this->zoom,
            'lock_wheel' => $this->lock_wheel ? true : false,
            'show_picture' => $this->show_picture ? true : false,
            'show_address' => $this->show_address ? true : false,
            'show_tel' => $this->show_tel ? true : false,
            'show_email' => $this->show_email ? true : false,
            'name' => $name,
            'address' => $this->address,
            'tel' => $this->tel,
            'email' => $this->email,
            't_address' => t('Address:'),
            't_tel' => t('Tel:'),
            't_email' => t('Email:'),
            't_lang' => t('en'),
            'picture' => $picture,
        );
        $this->set('data', json_encode($data, JSON_UNESCAPED_UNICODE));
        
        $this->set("show_map", $this->google_api_key == '' ? false : true);
    }

    public function add()
    {
        $this->set('servers', $this->getServers());
        $this->set('languages', $this->getLanguages());
        $this->set('latitude', '55.753994');
        $this->set('longitude', '37.622093');
        $this->set('pic', null);
    }

    public function edit()
    {
        $this->set('servers', $this->getServers());
        $this->set('languages', $this->getLanguages());
        $this->set('selected_server', $this->server);
        $this->set('selected_language', $this->language);
        
        $pic = null;
        if ($this->getPictureID() > 0) {
            $pic = $this->getPictureObject();
        }
        $this->set('pic', $pic);
    }
    
    public function save($args) 
    {
        $args['google_api_key'] = isset($args['google_api_key']) ? $args['google_api_key'] : '';
        $args['locationiq_api_key'] = isset($args['locationiq_api_key']) ? $args['locationiq_api_key'] : '';
        $args['server'] = isset($args['server']) ? $args['server'] : '';
        $args['language'] = isset($args['language']) ? $args['language'] : '';
        $args['name'] = isset($args['name']) ? $args['name'] : $args['location'];
        $args['address'] = isset($args['address']) ? $args['address'] : '';
        $args['tel'] = isset($args['tel']) ? $args['tel'] : '';
        $args['email'] = isset($args['email']) ? $args['email'] : '';
        $args['location'] = isset($args['location']) ? $args['location'] : '';
        $args['latitude'] = isset($args['latitude']) ? $args['latitude'] : '55.753994';
        $args['longitude'] = isset($args['longitude']) ? $args['longitude'] : '37.622093';
        $args['height'] = isset($args['height']) ? $args['height'] : '400px';
        $args['width'] = isset($args['width']) ? $args['width'] : '100%';
        $args['zoom'] = isset($args['zoom']) ? $args['zoom'] : '12';
        $args['lock_wheel'] = isset($args['lock_wheel']) ? 1 : 0;
        $args['show_geocoding'] = isset($args['show_geocoding']) ? 1 : 0;
        $args['pictureID'] = (isset($args['show_picture']) && $args['pictureID'] != 0) ? $args['pictureID'] : 0;
        $args['show_name'] = (isset($args['show_name']) && $args['name'] != '') ? 1 : 0;
        $args['show_address'] = (isset($args['show_address']) && $args['address'] != '') ? 1 : 0;
        $args['show_tel'] = (isset($args['show_tel']) && $args['tel'] != '') ? 1 : 0;
        $args['show_email'] = (isset($args['show_email']) && $args['email'] != '') ? 1 : 0;
        $args['show_picture'] = (isset($args['show_picture']) && $args['pictureID'] != 0) ? 1 : 0;
        parent::save($args);
    }

    public function getPictureID()
    {
        return isset($this->record->pictureID) ? $this->record->pictureID : (isset($this->pictureID) ? $this->pictureID : null);
    }

    public function getPictureObject()
    {
        return File::getByID($this->getPictureID());
    }

    public function getServers() {
        $servers = array(
            'eu1' => "Europe", 
            'ap1' => "Asia Pacific", 
            'us1' => "US East", 
            'us2' => "US West",
        );
        
        return $servers;
    }
     
    public function getLanguages() {
        $languages = array(
            'en' => "English", 
            'ru' => "Russian",
        );
        
        return $languages;
    }
     
}
