function SudokuInit (bID) {
    this.bID = bID;
    this.puzzle = new Sudoku();
    this.$container = $('#sudoku' + bID);
    this.$resultArea = this.$container.find('.solve');
    this.$newGameBtn = this.$container.find('button[name="newGame"]');
    this.$selectedCell = null;
}
 
SudokuInit.prototype.initAndStart = function () {
    var me = this;
    // initialize each cell.
    for (var i = 0; i < 9; i++) {
        for (var j = 0; j < 9; j++) {
            var $cell = this.getCell(i, j);

            // install the onclick handler
            $cell.click(function(e){
                me.selectCell($(this));
            });

            // when the page is first loaded, in firefox, first
            // child is an empty text node, in IE first child is
            // null.
            if (!$cell[0].firstChild) {
                $cell[0].appendChild(document.createTextNode(""));
            }

            // if the value is 0, create a blank cell.
            if (this.puzzle.getVal(i, j) == 0) {
                $cell[0].firstChild.nodeValue = "";
            } else {
                // if the value is not 0, set the value and mark
                // the cell as a hint.
                $cell[0].firstChild.nodeValue = this.puzzle.getVal(i, j);
                $cell.addClass("hint");
            }
        }
    }
    
    var me = this;
    $(document).on('keydown', this.$container, function (e) {
        me.getKey(e);
        if([32, 37, 38, 39, 40].indexOf(e.keyCode) > -1) {
            e.preventDefault();
        }
    });

    me.$container.on('click', '[data-numbers-val]', function () {
        me.buttonPressed($(this));
    });

    me.$container.on('click', '.close, button[name="ok"]', function () {
        me.newGame();
    });
    
    if(me.$newGameBtn.length > 0) {
        me.$newGameBtn.click(function(){
            me.newGame();
        });
    }

    me.$container.on('click', 'button[name="solveGame"]', function () {
        me.solve();
    });

    this.newGame();
};

// start a new game. this resets the board and generates a new puzzle.
SudokuInit.prototype.newGame = function () {
    var me = this;
    
    this.$resultArea.hide();
    
    if (this.$newGameBtn.length > 0) {
        this.$newGameBtn[0].disabled = true;
    }
        
    var $overlay = this.$container.find('.overlay');
    $overlay.show();

    this.unselectCell();

    this.puzzle.level = this.getSelectedLevel();
    this.puzzle.done = function () {
        // update the board with the new puzzle data.
        for (var i = 0; i < 9; i++) {
            for (var j = 0; j < 9; j++) {
                var $cell = me.getCell(i, j);
                $cell.removeClass("error");
                if (me.puzzle.getVal(i, j) == 0) {
                    $cell.removeClass("hint");
                    $cell[0].firstChild.nodeValue = "";
                } else {
                    $cell[0].firstChild.nodeValue = me.puzzle.getVal(i, j);
                    $cell.addClass("hint");
                }
            }
        }
        
        if (me.$newGameBtn.length > 0) {
            me.$newGameBtn[0].disabled = false;
        }
            
        $overlay.hide();
    };

    // generate the new puzzle.
    this.puzzle.newGame();
};

SudokuInit.prototype.getCell = function (i, j) {
    return this.$container.find('[data-cell="x' + i + "_" + j + '"]');
};

SudokuInit.prototype.getSelectedLevel = function () {
    var $input = this.$container.find('input[name="level"]');
    if ($input.prop("type") == "radio") {
        $input = $input.filter(':checked');
    }
    return parseInt($input.val());
};

// solve the game and display the solution.
SudokuInit.prototype.solve = function () {
    this.puzzle.solveGame();

    for (var i = 0; i < 9; i++) {
        for (var j = 0; j < 9; j++) {
            var $cell = this.getCell(i, j);
            $cell[0].firstChild.nodeValue = this.puzzle.getVal(i, j);

            this.showErrors(i, j); // this will actually remove the error
            // highlighting if there were errors
        }
    }
};

// selects the cell clicked on by the user.
SudokuInit.prototype.selectCell = function ($cell) {
    this.unselectCell();

    // if the cell is one that was automatically populated just return and
    // not allow the cell to be selected.
    if ($cell.hasClass("hint")) {
        return;
    }

    // save the selected cell and highlight the square on the board.
    $cell.addClass("selected");
    this.$selectedCell = $cell;
};

SudokuInit.prototype.unselectCell = function () {
    if (this.$selectedCell) {
        this.$selectedCell.removeClass("selected");
    }

    this.$selectedCell = null;
}

// checks the row, column and subsquare for the given row and column for any
        // conflicting values and highlights them. if there were any previously
        // highlighted that are no longer conflicting, the highlighting is removed.
SudokuInit.prototype.showErrors = function (row, col) {
    for (var i = 0; i < 9; i++)
    {
        var $cell = this.getCell(row, i);
        var val = this.puzzle.getVal(row, i);
        if (this.puzzle.checkVal(row, i, val) == true) {
            $cell.removeClass("error");
        } else {
            $cell.addClass("error");
        }
    }

    for (var i = 0; i < 9; i++)
    {
        var $cell = this.getCell(i, col);
        var val = this.puzzle.getVal(i, col);
        if (this.puzzle.checkVal(i, col, val) == true) {
            $cell.removeClass("error");
        } else {
            $cell.addClass("error");
        }
    }

    var r = row - row % 3;
    var c = col - col % 3;
    for (var i = r; i < r + 3; i++) {
        for (var j = c; j < c + 3; j++) {
            var $cell = this.getCell(i, j);
            var val = this.puzzle.getVal(i, j);
            if (this.puzzle.checkVal(i, j, val) == true) {
                $cell.removeClass("error");
            } else {
                $cell.addClass("error");
            }
        }
    }
};

// move the selected cell up to the next user selectable cell. if no cells are
// selected this will automatically select the lowest right-most user selectable
// cell. if there are no more cells up in the current column the selection will
// wrap to the bottom and move one column to the left.
SudokuInit.prototype.moveUpSelectedCell = function () {
    var row;
    var col;
    if (!this.$selectedCell) {
        row = 9;
        col = 8;
    } else {
        var id = this.$selectedCell.attr('data-cell');
        id = id.substr(1);
        var arr = id.split("_");
        row = arr[0];
        col = arr[1];
    }

    this.unselectCell();

    var $cell;
    do {
        row--;
        if (row < 0) {
            row = 8;
            col = (col + 8) % 9;
        }
        $cell = this.getCell(row, col);
    } while ($cell.hasClass("hint"));

    this.$selectedCell = $cell;
    this.$selectedCell.addClass("selected");
};

// move the selected cell down to the next user selectable cell. if no cells are
// selected this will automatically select the top left-most user selectable
// cell. if there are no more cells down in the current column the selection
// will wrap to the top and move one column to the right.
SudokuInit.prototype.moveDownSelectedCell = function () {
    var row;
    var col;

    if (!this.$selectedCell) {
        row = -1;
        col = 0;
    } else {
        var id = this.$selectedCell.attr('data-cell');
        id = id.substr(1);
        var arr = id.split("_");
        row = arr[0];
        col = arr[1];
    }

    this.unselectCell();

    var $cell;
    do {
        row++;
        if (row > 8) {
            row = 0;
            col = (col + 1) % 9;
        }
        $cell = this.getCell(row, col);
    } while ($cell.hasClass("hint"));

    this.$selectedCell = $cell;
    this.$selectedCell.addClass("selected");
};

// move the selected cell left to the next user selectable cell. if no cells are
// selected this will automatically select the right-most bottom user selectable
// cell. if there are no more cells left in the current row the selection will
// wrap to the right and move one row up.
SudokuInit.prototype.moveLeftSelectedCell = function () {
    var row;
    var col;
    if (!this.$selectedCell) {
        row = 8;
        col = 9;
    } else {
        var id = this.$selectedCell.attr('data-cell');
        id = id.substr(1);
        var arr = id.split("_");
        row = arr[0];
        col = arr[1];
    }

    this.unselectCell();

    var $cell;
    do {
        col--;
        if (col < 0) {
            col = 8;
            row = (row + 8) % 9;
        }
        $cell = this.getCell(row, col);
    } while ($cell.hasClass("hint"));

    this.$selectedCell = $cell;
    this.$selectedCell.addClass("selected");
}

// move the selected cell right to the next user selectable cell. if no cells
// are selected this will automatically select the left-most top user
// selectable cell. if there are no more cells right in the current row the
// selection will wrap to the left and move one row down.
SudokuInit.prototype.moveRightSelectedCell = function () {
    var row;
    var col;
    if (!this.$selectedCell) {
        row = 0;
        col = -1;
    } else {
        var id = this.$selectedCell.attr('data-cell');
        id = id.substr(1);
        var arr = id.split("_");
        row = arr[0];
        col = arr[1];
    }

    this.unselectCell();

    var $cell;
    do {
        col++;
        if (col > 8) {
            col = 0;
            row = (row + 1) % 9;
        }
        $cell = this.getCell(row, col);
    } while ($cell.hasClass("hint"));

    this.$selectedCell = $cell;
    this.$selectedCell.addClass("selected");
}

// sets the value for the selected cell.
SudokuInit.prototype.setVal = function (row, col, val) {
    // if there is no cell selected, ignore the input value.
    if (!this.$selectedCell)
        return;

    // set the puzzle value and draw the value in the cell.
    this.puzzle.setVal(1 * row, 1 * col, val);
    this.$selectedCell[0].firstChild.nodeValue = (val > 0) ? val : "";

    // check for conflicting values according to the sudoku rules and mark
    // them.
    this.showErrors(1 * row, 1 * col);

    // check to see if the game is done.
    if ((val = this.puzzle.gameFinished()) != 0)
    {
        this.unselectCell();
        var h = Math.floor(val / (60 * 60 * 1000));
        var m = Math.floor(val % (60 * 60 * 1000) / (60 * 1000));
        var s = Math.floor(val % (60 * 60 * 1000) % (60 * 1000) / 1000);
        this.$container.find('.solve-h').html(h);
        this.$container.find('.solve-m').html(m);
        this.$container.find('.solve-s').html(s);
        this.$resultArea.show();
    }
}

// gets the keyboard input.
SudokuInit.prototype.getKey = function (e) {
    var id;
    var arr;
    if (this.$selectedCell)
    {
        id = this.$selectedCell.attr('data-cell');
        id = id.substr(1);
        arr = id.split("_");
    }

    var code;
    if (e.keyCode)
        code = e.keyCode;
    else if (e.which)
        code = e.which;

    switch (code)
    {
        case 37: // lt
            this.moveLeftSelectedCell();
            break;
        case 38: // up
            this.moveUpSelectedCell();
            break;
        case 39: // rt
            this.moveRightSelectedCell();
            break;
        case 40: // dn
            this.moveDownSelectedCell();
            break;
        case 8: // backspace
        case 46: // delete
            if (arr)
                this.setVal(arr[0], arr[1], 0);
            return false;
            break;
        case 27: // escape
            this.unselectCell();
            break;
        default:
            if (arr)
            {
                if (code >= 49 && code <= 57) {
                    this.setVal(arr[0], arr[1], code - 48);
                } else if (code >= 97 && code <= 105) {
                    this.setVal(arr[0], arr[1], code - 96);
                }
            }
            break;
    }
};

SudokuInit.prototype.buttonPressed = function ($btn) {
    var id;
    var arr;
    if (this.$selectedCell) {
        id = this.$selectedCell.attr('data-cell');
        id = id.substr(1);
        arr = id.split("_");
        this.setVal(arr[0], arr[1], $btn.attr('data-numbers-val'));
    }
};

// implementing trim because IE problems.
if(typeof String.prototype.trim !== 'function') {
    String.prototype.trim = function() {
        return this.replace(/^\s\s*/, '').replace(/\s\s*$/, '');
    }
}